package gov.va.med.mhv.usermgmt.service.impl;

import java.sql.Timestamp;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.security.auth.login.LoginException;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.config.ConfigurationManager;
import org.tigris.atlas.messages.CoreMessages;
import org.tigris.atlas.service.BooleanServiceResponse;
import org.tigris.atlas.service.ServiceResponse;
import org.tigris.atlas.service.StringServiceResponse;
import org.tigris.atlas.service.VoidServiceResponse;

import com.bea.p13n.security.Authentication;

import gov.va.med.mhv.core.service.delegate.EmailServiceDelegate;
import gov.va.med.mhv.service.MHVAbstractService;
import gov.va.med.mhv.usermgmt.bizobj.UserPasswordHistoryBO;
import gov.va.med.mhv.usermgmt.enumeration.ActivityActorTypeEnumeration;
import gov.va.med.mhv.usermgmt.messages.UserManagementMessages;
import gov.va.med.mhv.usermgmt.persist.DaoFactory;
import gov.va.med.mhv.usermgmt.persist.DuplicateEntryException;
import gov.va.med.mhv.usermgmt.persist.NoSuchEntryException;
import gov.va.med.mhv.usermgmt.persist.UserEntryDao;
import gov.va.med.mhv.usermgmt.service.EmployeeSearchResultServiceResponse;
import gov.va.med.mhv.usermgmt.service.ServiceFactory;
import gov.va.med.mhv.usermgmt.service.UserProfileService;
import gov.va.med.mhv.usermgmt.service.UserProfileServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserService;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.util.AccountSecurityManager;
import gov.va.med.mhv.usermgmt.util.Auditor;
import gov.va.med.mhv.usermgmt.util.MessageKeys;
import gov.va.med.mhv.usermgmt.util.PasswordUtil;

/**
 * Service implementation class for the User service
 */
public class UserServiceImpl extends MHVAbstractService implements UserService {

	private static final Log LOG = LogFactory.getLog(UserServiceImpl.class);

	private static final Properties PROPS = ConfigurationManager.getConfiguration("/mailCfg.properties");
	private static final String EMAIL_SUBJECT = "My HealtheVet Account Activity Update";

	private UserEntryDao userDao;

	private static Set<Character> SPECIAL_CHARS = createSpecialChars();

	private static final Set<Character> createSpecialChars() {
		Set<Character> specialChars = new HashSet<Character>( 32);
		specialChars.add( new Character( '!' ) );
		specialChars.add( new Character( '@' ) );
		specialChars.add( new Character( '#' ) );
		specialChars.add( new Character( '$' ) );
		specialChars.add( new Character( '%' ) );
		specialChars.add( new Character( '&' ) );
		specialChars.add( new Character( '*' ) );
		specialChars.add( new Character( '|' ) );
		specialChars.add( new Character( '?' ) );
		specialChars.add( new Character( '.' ) );
		specialChars.add( new Character( '_' ) );
		specialChars.add( new Character( '/' ) );
		specialChars.add( new Character( '<' ) );
		specialChars.add( new Character( '>' ) );
		specialChars.add( new Character( ';' ) );
		specialChars.add( new Character( ':' ) );
		specialChars.add( new Character( '"' ) );
		specialChars.add( new Character( '\\' ) );
		specialChars.add( new Character( '=' ) );
		specialChars.add( new Character( '+' ) );
		specialChars.add( new Character( '-' ) );
		specialChars.add( new Character( '`' ) );
		specialChars.add( new Character( '~' ) );
		specialChars.add( new Character( ')' ) );
		specialChars.add( new Character( '(' ) );
		specialChars.add( new Character( ',' ) );
		specialChars.add( new Character( '\'' ) );
		specialChars.add( new Character( '{' ) );
		specialChars.add( new Character( '}' ) );
		//CR1256 to add ^ [ and ] as special characters
		specialChars.add( new Character( '^' ) );
		specialChars.add( new Character( '[' ) );
		specialChars.add( new Character( ']' ) );
		return specialChars;
	}

    @Override
    protected Log getLog() {
        return LOG;
    }

	/**
	 * Execute the RemoveUser service
	 *
	 * @return RemoveUserServiceResponse
	 */
	public VoidServiceResponse removeUser(String userName) {
		VoidServiceResponse response = new VoidServiceResponse();
		try {
			getUserDao().removeUser(userName);
		} catch (Throwable t) {
			LOG.info("Failed to remove user", t);
		}
		return response;
	}

	/**
	 * Execute the Authenticate service
	 *
	 * @return AuthenticateServiceResponse
	 */
	public BooleanServiceResponse authenticate(String userName,
		String password)
	{
		if (StringUtils.isBlank(userName) || StringUtils.isBlank(password)) {
			return createNamePasswordRequiredResponse();
		}
		UserProfile userProfile = findUserProfile(userName);
		if (userProfile == null) {
	    	return createInvalidCredentialsResponse();
		}
        if (isAccountDeactivated(userProfile)) {
            return createAccountDeactivatedResponse(userProfile);
        }
		if (isAccountLocked(userProfile)) {
			return createAccountLockedResponse();
		}
		if (hasAccountExpired(userProfile)) {
			return createPasswordExpiredResponse();
		}
		try {
			Authentication.authenticate(userName, password);
		} catch (LoginException e) {
	    	return isAccountLocked(userProfile)
	    		? createAccountLockedResponse()
    			: createInvalidCredentialsResponse();
		}
		resetAccountLock(userProfile);

		return createBooleanResponse(true);

	}

	public BooleanServiceResponse isAccountLocked(String userName) {
		UserProfile userProfile = findUserProfile(userName);
		return createBooleanResponse((userProfile != null)
					&& isAccountLocked(userProfile));
	}

	/**
	 * Execute the CreateUser service
	 *
	 * @return CreateUserServiceResponse
	 */
	public VoidServiceResponse createUser(String userName, String password,
		String confirmPassword)
	{
		VoidServiceResponse response = new VoidServiceResponse();

		validateUserCredentials(userName, password, confirmPassword, response);
		if( response.getMessages().hasErrorMessages() ) {
			return response;
		}

        // This is needed as authentication is done against both the
        // User (LDAP) and Employee (Active Directory) authentication
        // providers (most like against LDAP first).
        // Not allowing users to pick a user name that is in use by an
        // employee, avoids user name conflicts upon authentication, and
        // subsequently possible user hijacking (see CR 2018).
        // NOTE: This check is snapshot and does not account for employee
        // accounts created after the snapshot.
		Boolean userAlreadyExists = existsEmployeeWithUserName(userName);
        if (userAlreadyExists == null) {
            // Unable to establish whether employee exists, e.g.
            // Active Directory is unavailable
            addError(response, MessageKeys.UNKNOWN_EXCEPTION_OCCURRED, null);
        } else {
            // Now autoboxing will work properly on userAlreadyExists
            if (!userAlreadyExists) {
    			try {
    				getUserDao().createUser(userName, password);
    			} catch(DuplicateEntryException e) {
    				userAlreadyExists = true;
    			}
    		}
    		if (userAlreadyExists) {
    			addError(response, UserManagementMessages.USER_ALREADY_EXISTS,
                    new String[]{userName});
    		}
        }

		return response;
	}

	private Boolean existsEmployeeWithUserName(String userName) {
		Boolean exists = null;
		EmployeeSearchResultServiceResponse response =
			ServiceFactory.createEmployeeSearchService().searchByUserName(
			userName);
		if (!response.getMessages().hasErrorMessages()) {
			exists = (response.getEmployeeSearchResult() != null);
		}
		return exists;
	}

	private void validateUserCredentials(String userName, String password, String confirmPassword, ServiceResponse response) {
		validateUserName( userName, response );
		validatePassword( userName, password, confirmPassword, response );
	}

	private void validateUserName(String userName, ServiceResponse response ) {

		if( StringUtils.isBlank( userName ) ) {
			addError( response, CoreMessages.NULL_NOT_ALLOWED, new Object[] { "User name" } );
		}
		else {
			if( userName.length() < 6 || userName.length() > 12 ) {
				addError( response, UserManagementMessages.USERNAME_INVALID_LENGTH );
			}

			if( ! StringUtils.isAlphanumeric( userName ) ) {
				addError( response, UserManagementMessages.USERNAME_NOT_ALPHANUMERIC );
			}
		}

	}

	private void validatePassword(String userName, String password, String confirmPassword, ServiceResponse response) {
		boolean blankValues = false;
		if( StringUtils.isBlank( password ) ) {
			addError(response, CoreMessages.NULL_NOT_ALLOWED, new Object[] { "Password" } );
			blankValues = true;
		}
		if( StringUtils.isBlank( confirmPassword ) ) {
			addError(response, CoreMessages.NULL_NOT_ALLOWED, new Object[] { "Confirm password" } );
			blankValues = true;
		}
		if( !password.equals( confirmPassword ) ) {
			addError(response, UserManagementMessages.PASSWORD_NOT_CONFIRM);
			blankValues = true;
		}
		if( blankValues ) {
			return;
		}

		if( password.equals( userName ) ) {
			addError( response, UserManagementMessages.USERNAME_PASSWORD_EQUAL );

		}

		if( password.length() < 8 || password.length() > 12 ) {
			addError( response, UserManagementMessages.PASSWORD_INVALID_LENGTH );

		}

		if( StringUtils.contains( password, " " ) ) {
			addError( response, UserManagementMessages.PASSWORD_SPACES_NOT_ALLOWED );

		}

		boolean hasSpecialChar = false;
		boolean hasDigit = false;
		boolean hasLetter = false;
		for( int i=0; i<password.length(); i++ ) {
			if( hasSpecialChar && hasDigit && hasLetter ) {
				break;
			}
			char c = password.charAt( i );
			if( !hasLetter && Character.isLetter( c ) ) {
				hasLetter = true;
			}
			else if( !hasDigit && Character.isDigit( c ) ) {
				hasDigit = true;
			}
			else if( !hasSpecialChar && SPECIAL_CHARS.contains( new Character( c ) ) ) {
				hasSpecialChar = true;
			}
		}
		if( ! hasSpecialChar ) {
			addError( response, UserManagementMessages.PASSWORD_NO_SPECIAL_CHAR );

		}
		if( ! hasLetter ) {
			addError( response, UserManagementMessages.PASSWORD_NO_LETTER );

		}
		if( ! hasDigit ) {
			addError( response, UserManagementMessages.PASSWORD_NO_DIGIT );

		}

	}


	/**
	 * Change the password of the user identified by the userName parameter.
	 * @param userName
	 * @param currentPassword
	 * @param newPassword
	 * @param confirmNewPassword
	 * @return ChangePasswordServiceResponse
	 */
	public VoidServiceResponse changePassword(String userName, String currentPassword, String newPassword, String confirmNewPassword) {

		VoidServiceResponse response = new VoidServiceResponse();

		UserProfileService userProfileService = ServiceFactory.createUserProfileService();
	    UserProfile userProfile = userProfileService.getProfileForUser(userName).getUserProfile();

		//if((currentPassword != null)) {
			BooleanServiceResponse authenticationResponse = authenticate(
				userName, currentPassword);
			if (authenticationResponse.getMessages().hasErrorMessages()) {
				copyMessagesOnError(response, authenticationResponse);
				return response;
			}
		//}

		validatePassword( userName, newPassword, confirmNewPassword, response );

		//check if the password has been used in the last three passwords
		BooleanServiceResponse boolResponse=userProfileService.hasPasswordBeenUsed(
			userProfile, newPassword);
		if (boolResponse.getMessages().hasErrorMessages()){
			response.getMessages().addMessages( boolResponse.getMessages());
		}
		if (boolResponse.getBoolean().booleanValue()){
			addError( response, UserManagementMessages.PASSWORD_USED);
		}

		//	end of check if the password has been used in the last three passwords
		if (response.getMessages().hasErrorMessages()) {
			return response;
		}

		//now save the password in the password history
		userProfileService = ServiceFactory.createUserProfileService();
		VoidServiceResponse voidResponse = userProfileService.
			savePasswordHistory(userProfile, newPassword, Boolean.FALSE);
		if (voidResponse.getMessages().hasErrorMessages()){
			response.getMessages().addMessages( voidResponse.getMessages());
		}
		resetAccountLock(userProfile);

		try {
			getUserDao().resetPassword(userName, newPassword);
		} catch(NoSuchEntryException e) {
			//add username to the error message
			addError(response,UserManagementMessages.USER_DOES_NOT_EXIST, new String[]{userName});
		}

		//Auditor.auditChangePasswordEvent(userProfile.getId(), ActivityActorTypeEnumeration.SELF, success);


		/*
		 * CR2897: Sends out an email once the user changes the password successfully (only if the user has an email address in the profile)
		 */
		String userEmail = userProfileService.getProfileForUser(userName).getUserProfile().getContactInfoEmail();

	    EmailServiceDelegate service = gov.va.med.mhv.core.service.delegate.ServiceDelegateFactory.createEmailServiceDelegate();

		if (!StringUtils.isBlank(userEmail)) {
			/*
			 * CR 3350; replyTo address has been changed and is now coming from email config properties file
			 */
			//service.sendMessage(PROPS.getProperty("userprofile.from.email"), EMAIL_SUBJECT, getMessage(), userEmail.toString());
			service.sendMessageWithReplyTo(PROPS.getProperty("userprofile.from.email"), EMAIL_SUBJECT, getMessage(), userEmail.toString(), PROPS.getProperty("userprofile.replyTo.email"));
		}

		return response;
	}



	/**
	 * Reset the password of the user identified by the userName parameter.
	 * Save the password history and set the temp flag to true
	 * @param userName
	 * @param newPassword
	 * @param confirmNewPassword
	 * @return ResetPasswordServiceResponse
	 */
	public VoidServiceResponse resetPassword(String userName,
		String newPassword, String confirmNewPassword, Boolean isTemporary)
	{
		VoidServiceResponse response = new VoidServiceResponse();

		validatePassword( userName, newPassword, confirmNewPassword, response );

		UserProfileService userProfileService = ServiceFactory.
			createUserProfileService();
		UserProfileServiceResponse userProfileResponse = userProfileService.
			getProfileForUser(userName);
		copyMessagesOnError(response, userProfileResponse);
		if(response.getMessages().hasErrorMessages()) {
			return response;
		}
		UserProfile userProfile = userProfileResponse.getUserProfile();

		//check if the password has been used in the last three passwords
		BooleanServiceResponse boolResponse= userProfileService.
			hasPasswordBeenUsed(userProfile, newPassword);
        copyMessagesOnError(response, boolResponse);
		if (boolResponse.getBoolean().booleanValue()){
			addError( response, UserManagementMessages.PASSWORD_USED);
		}
		//end of check if the password has been used in the last three passwords
		if(response.getMessages().hasErrorMessages()) {
			return response;
		}

		VoidServiceResponse voidResponse = userProfileService.
			savePasswordHistory(userProfile, newPassword, isTemporary);
		copyMessagesOnError(response, voidResponse);
		resetAccountLock(userProfile);

		try {
			getUserDao().resetPassword(userName, newPassword);
		} catch(NoSuchEntryException e) {
			addError(response,UserManagementMessages.USER_DOES_NOT_EXIST,
                new String[]{userName});
		}

		//CCR#3233: per email from Lisa Schope,
		//When the help desk generates a password through the admin portal it is temporary
		//and the user is forced to change it when entering MHV.
		//When the user uses the forgot password function it is not temporary.
		if (isTemporary.booleanValue())
		{
			Auditor.auditResetPasswordEvent(userProfile.getId(),
					ActivityActorTypeEnumeration.HELP_DESK_ADMINISTRATOR);
		} else
		{
			Auditor.auditResetPasswordEvent(userProfile.getId(),
					ActivityActorTypeEnumeration.SELF);
		}

		return response;
	}

	/*
	 * Builds an email body for change password and forgot password portlets
	 */
	private static String getMessage() {
	    StringBuilder messageText = new StringBuilder();

	    messageText.append("This is a courtesy message to notify you of recent activity involving your My HealtheVet password or User ID." +
	    		           " If you did not recently change your password or request your User ID, please notify us by using the Contact MHV" +
	    		           " link provided on the My HealtheVet website - " + "www.myhealth.domain.");

	    messageText.append("\n\n ------------------------------------------------------- ");
	    messageText.append("\n **This is an automated message.  Please do not reply** ");
	    messageText.append("\n ------------------------------------------------------- \n\n");

	    messageText.append("This e-mail is intended only for the person or entity to which it is addressed, and may contain information that" +
	    		           " is privileged, confidential, or otherwise protected from disclosure.  Dissemination, distribution, or copying of" +
	    		           " this e-mail or the information herein by anyone other than the intended recipient or for official internal VHA " +
	    		           " and/or VA business is prohibited.");

	    messageText.append("\n\n");

	    return messageText.toString();
	  }




	/**
	 * Change a user's username.
	 * @param currentUserName
	 * @param newUserName
	 * @param newPassword
	 * @param confirmNewPassword
	 * @return ChangeUserNameServiceResponse
	 */
	public VoidServiceResponse changeUserName(String currentUserName, String newUserName, String newPassword, String confirmNewPassword) {
		VoidServiceResponse response = new VoidServiceResponse();

		try {
			getUserDao().changeUserName(currentUserName, newUserName);
		}
		catch(NoSuchEntryException e) {
			addError(response, UserManagementMessages.USER_DOES_NOT_EXIST);
		}

		return response;
	}

	public StringServiceResponse generatePassword() {
		StringServiceResponse response = new StringServiceResponse();
		String tempPassword = PasswordUtil.generateUserPassword();
		response.setValue(tempPassword);
		return response;
	}

	protected UserEntryDao getUserDao() {
		if(userDao == null)
			userDao = (UserEntryDao) DaoFactory.createDao(UserEntryDao.BEAN_NAME);

		return userDao;
	}


	private UserProfile findUserProfile(String userName) {
		if (StringUtils.isBlank(userName)) {
			return null;
		}
		userName = StringUtils.lowerCase(userName);
		UserProfileServiceResponse response = ServiceFactory.
	       	createUserProfileService().getProfileForUser(userName);
	    if (response.getMessages().hasErrorMessages()) {
	    	// System error: Propagate up to have the messages log with a
	    	// stack trace and alert the user with generic message
	    	throw new IllegalStateException("" + response.getMessages().
	    		getErrorMessages());
        }
		return response.getUserProfile();
	}

	private boolean isAccountLocked(UserProfile userProfile) {
		assert userProfile != null : "Missing user profile";
		return AccountSecurityManager.getInstance().isAccountLocked(
			userProfile.getUserName());
	}

	/**
	 * Resets the account lock count and date.
	 * Should be called on successful authentication, as the lock count is
	 * used to count the number of subsequent failures and the lock date is
	 * used to denote the start of the lock (locks last a fixed amount of
	 * time, e.g. 30 minutes).
	 * @param userProfile The user profile to reset the lock information for.
	 */
	private void resetAccountLock(UserProfile userProfile) {
		assert userProfile != null : "Missing userProfile";
		AccountSecurityManager.getInstance().unlockAccount(userProfile.
			getUserName());
	}

	private boolean hasAccountExpired(UserProfile userProfile) {
		assert userProfile != null : "Missing userProfile";
		BooleanServiceResponse response = ServiceFactory.
			createUserProfileService().isCurrentPasswordTemporary(userProfile);
		if (!response.getBoolean().booleanValue()) {
			return false; // not a temporary password, does not expire
		}

		List history = UserPasswordHistoryBO.getPasswordHistoryByUserProfileId(
			userProfile.getId());
		assert history != null : "UserProfile for user '"
			+ userProfile.getUserName() + "' has no password history.";
		UserPasswordHistoryBO password = (UserPasswordHistoryBO) history.get(0);
		assert password != null : "User '" + userProfile.getUserName()
			+ "' has no current password.";
		Timestamp updateTime = password.getUpdateTime();
		assert updateTime != null : "No update time found for current "
			+ "password of user '" + userProfile.getUserName() + "'";

		return AccountSecurityManager.getInstance().hasAccountExpired(
			updateTime);
	}

    private boolean isAccountDeactivated(UserProfile userProfile) {
        return (userProfile.getDeactivationReason() != null);
    }

	private BooleanServiceResponse createInvalidCredentialsResponse() {
		BooleanServiceResponse response = createBooleanResponse(false);
		addError(response, UserManagementMessages.INVALID_CREDENTIALS, null);
		return response;
	}

    private BooleanServiceResponse createAccountDeactivatedResponse(
        UserProfile userProfile)
    {
        BooleanServiceResponse response = createBooleanResponse(false);
        addError(response, UserManagementMessages.ACCOUNT_DEACTIVATED);
        addReferenceId(response, userProfile);
        return response;
    }

    private void addReferenceId(ServiceResponse response,
        UserProfile userProfile)
    {
        if ((userProfile != null) && userProfile.getIsPatient()) {
//            PatientServiceResponse patientResponse = ServiceFactory.
//                createPatientService().getPatientForUser(userProfile);
//            if (hasErrorMessages(patientResponse)) {
//                logMessages(patientResponse);
//            } else if (patientResponse.getPatient() == null) {
//                LOG.error("Patient record not found for user '" + userProfile.
//                    getUserName() + "'");
//            } else {
//                String referenceId = patientResponse.getPatient().
//                    getInvalidationReferenceId();
//                if (!StringUtils.isBlank(referenceId))
//                   addError(response, UserManagementMessages.
//                       ACCOUNT_DEACTIVATED,new Object[] { referenceId });
//            }
        }
    }

	private BooleanServiceResponse createAccountLockedResponse() {
		BooleanServiceResponse response = createBooleanResponse(false);
		addError(response, UserManagementMessages.ACCOUNT_LOCKED,
			 new Object[] {
				AccountSecurityManager.getInstance().getLockoutPeriod()
			});
		return response;
	}

	private BooleanServiceResponse createNamePasswordRequiredResponse() {
		BooleanServiceResponse response = createBooleanResponse(false);
		addError(response, UserManagementMessages.USERNAME_PASSWORD_REQUIRED,
			null);
		return response;
	}

	private BooleanServiceResponse createPasswordExpiredResponse() {
		BooleanServiceResponse response = createBooleanResponse(false);
		addError(response, UserManagementMessages.CURRENT_PASSWORD_EXPIRED,
			null);
		return response;
	}

	private BooleanServiceResponse createBooleanResponse(boolean value) {
		BooleanServiceResponse bResponse = new BooleanServiceResponse();
		bResponse.setValue(value);
		return bResponse;
	}

}
